"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.slice = slice;
exports.offsetAndLimit = offsetAndLimit;
exports.parseLimit = parseLimit;
exports.filterPredicate = filterPredicate;
exports.getWithScoresAndLimit = getWithScoresAndLimit;

function normalizeIndex(index, array) {
  if (index < 0) {
    return -index > array.length ? 0 : array.length + index;
  }

  return index;
}

function slice(arr, s, e) {
  var start = normalizeIndex(s, arr);
  var end = normalizeIndex(e, arr);

  if (end === -1) {
    return [];
  }

  if (end - start < 0) {
    return Array.from(arr).reverse().slice(start, end + 1);
  }

  return arr.slice(start, end + 1);
}

function normalizeCountToIndex(offset, count, array) {
  if (count < 0) {
    return -count > array.length ? 0 : array.length + count;
  }

  return offset + count;
}

function offsetAndLimit(arr, offset, count) {
  if (count === 0) {
    return [];
  }

  var end = normalizeCountToIndex(offset, count, arr);
  return arr.slice(offset, end);
}

function parseLimit(input) {
  var str = "".concat(input);
  var strict = false;

  if (str[0] === '(') {
    str = str.substr(1, str.length);
    strict = true;
  } else if (str === '-inf') {
    return {
      value: -Infinity,
      isStrict: true
    };
  } else if (str === '+inf') {
    return {
      value: +Infinity,
      isStrict: true
    };
  }

  return {
    value: parseInt(str, 10),
    isStrict: strict
  };
}

function filterPredicate(min, max) {
  return function (it) {
    if (it.score < min.value || min.isStrict && it.score === min.value) {
      return false;
    }

    if (it.score > max.value || max.isStrict && it.score === max.value) {
      return false;
    }

    return true;
  };
} // this emulates the way Redis handles parsing these arguments
// see https://github.com/antirez/redis/blob/06d490342f51cff316588a7a45124cc410b7d050/src/t_zset.c#L2556


function getWithScoresAndLimit(args) {
  var remaining = args.length;
  var pos = 0;
  var withScores = false;
  var limit = null;
  var offset = null;

  while (remaining > 0) {
    if (remaining >= 1 && args[pos].toUpperCase() === 'WITHSCORES') {
      withScores = true;
      pos += 1;
      remaining -= 1;
    } else if (remaining >= 3 && args[pos].toUpperCase() === 'LIMIT') {
      offset = parseInt(args[pos + 1], 10);
      limit = parseInt(args[pos + 2], 10);
      pos += 3;
      remaining -= 3;
    } else {
      throw new Error('ERR syntax error');
    }
  }

  return {
    withScores: withScores,
    limit: limit,
    offset: offset
  };
}