"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isInSubscriberMode = isInSubscriberMode;
exports.isNotConnected = isNotConnected;
exports.throwIfInSubscriberMode = throwIfInSubscriberMode;
exports.throwIfNotConnected = throwIfNotConnected;
exports.throwIfCommandIsNotAllowed = throwIfCommandIsNotAllowed;
exports.processArguments = processArguments;
exports.processReply = processReply;
exports.safelyExecuteCommand = safelyExecuteCommand;
exports.default = command;

var _lodash = _interopRequireDefault(require("lodash"));

var _standardAsCallback = _interopRequireDefault(require("standard-as-callback"));

var _promiseContainer = _interopRequireDefault(require("./promise-container"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _toConsumableArray(arr) { return _arrayWithoutHoles(arr) || _iterableToArray(arr) || _unsupportedIterableToArray(arr) || _nonIterableSpread(); }

function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }

function _unsupportedIterableToArray(o, minLen) { if (!o) return; if (typeof o === "string") return _arrayLikeToArray(o, minLen); var n = Object.prototype.toString.call(o).slice(8, -1); if (n === "Object" && o.constructor) n = o.constructor.name; if (n === "Map" || n === "Set") return Array.from(o); if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n)) return _arrayLikeToArray(o, minLen); }

function _iterableToArray(iter) { if (typeof Symbol !== "undefined" && Symbol.iterator in Object(iter)) return Array.from(iter); }

function _arrayWithoutHoles(arr) { if (Array.isArray(arr)) return _arrayLikeToArray(arr); }

function _arrayLikeToArray(arr, len) { if (len == null || len > arr.length) len = arr.length; for (var i = 0, arr2 = new Array(len); i < len; i++) { arr2[i] = arr[i]; } return arr2; }

function isInSubscriberMode(RedisMock) {
  if (RedisMock.channels === undefined) {
    return false;
  }

  return RedisMock.subscriberMode;
}

function isNotConnected(RedisMock) {
  if (RedisMock.connected === undefined) {
    return false;
  }

  return !RedisMock.connected;
}

function throwIfInSubscriberMode(commandName, RedisMock) {
  if (isInSubscriberMode(RedisMock)) {
    var allowedCommands = ['subscribe', 'psubscribe', 'unsubscribe', 'punsubscribe', 'ping', 'quit', 'disconnect'];

    if (allowedCommands.indexOf(commandName) > -1) {// command is allowed -> do not throw
    } else {
      throw new Error('Connection in subscriber mode, only subscriber commands may be used');
    }
  }
}

function throwIfNotConnected(commandName, RedisMock) {
  if (isNotConnected(RedisMock)) {
    if (commandName !== 'connect' && commandName !== 'defineCommand') {
      throw new Error("Stream isn't writeable and enableOfflineQueue options is false");
    }
  }
}

function throwIfCommandIsNotAllowed(commandName, RedisMock) {
  throwIfInSubscriberMode(commandName, RedisMock);
  throwIfNotConnected(commandName, RedisMock);
}
/**
 * Transform non-buffer arguments to strings to simulate real ioredis behavior
 * @param {any} arg the argument to transform
 */


var argMapper = function argMapper(arg) {
  if (arg === null || arg === undefined) return '';
  return arg instanceof Buffer ? arg : arg.toString();
};

function processArguments(args, commandName, RedisMock) {
  // fast return, the defineCommand command requires NO transformation of args
  if (commandName === 'defineCommand') return args;
  var commandArgs = args ? _lodash.default.flatten(args) : [];

  if (RedisMock.Command.transformers.argument[commandName]) {
    commandArgs = RedisMock.Command.transformers.argument[commandName](args);
  }

  commandArgs = commandArgs.map(argMapper);
  return commandArgs;
}

function processReply(result, commandName, RedisMock) {
  if (RedisMock.Command.transformers.reply[commandName]) {
    return RedisMock.Command.transformers.reply[commandName](result);
  }

  return result;
}

function safelyExecuteCommand(commandEmulator, commandName, RedisMock) {
  throwIfCommandIsNotAllowed(commandName, RedisMock);

  for (var _len = arguments.length, commandArgs = new Array(_len > 3 ? _len - 3 : 0), _key = 3; _key < _len; _key++) {
    commandArgs[_key - 3] = arguments[_key];
  }

  var result = commandEmulator.apply(void 0, commandArgs);
  return processReply(result, commandName, RedisMock);
}

function command(commandEmulator, commandName, RedisMock) {
  return function () {
    for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }

    var lastArgIndex = args.length - 1;
    var callback = args[lastArgIndex];

    if (typeof callback !== 'function') {
      callback = undefined;
    } else {
      // eslint-disable-next-line no-param-reassign
      args.length = lastArgIndex;
    }

    var commandArgs = processArguments(args, commandName, RedisMock);

    if (commandName === 'defineCommand') {
      return safelyExecuteCommand.apply(void 0, [commandEmulator, commandName, RedisMock].concat(_toConsumableArray(commandArgs)));
    }

    var Promise = _promiseContainer.default.get();

    return (0, _standardAsCallback.default)(new Promise(function (resolve) {
      return resolve(safelyExecuteCommand.apply(void 0, [commandEmulator, commandName, RedisMock].concat(_toConsumableArray(commandArgs))));
    }), callback);
  };
}